<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class PAAM_Settings {
	private $option_key = 'paam_options';

	public function __construct() {
		add_action( 'admin_menu', [ $this, 'menu' ] );
		add_action( 'admin_init', [ $this, 'register' ] );
	}

	public function menu() {
		add_options_page(
			__('Pro AMP AdSense', 'pro-amp-adsense'),
			__('Pro AMP AdSense', 'pro-amp-adsense'),
			'manage_options',
			'paam-settings',
			[ $this, 'render_page' ]
		);
	}

	public function register() {
		register_setting( 'paam_group', $this->option_key, [ $this, 'sanitize' ] );
	}

	public function get_options() {
		$opts = get_option( $this->option_key, [] );
		return is_array( $opts ) ? $opts : [];
	}

	public function sanitize( $input ) {
		$san = $this->get_options();
		$san['publisher_id'] = isset($input['publisher_id']) ? sanitize_text_field($input['publisher_id']) : '';
		$san['consent_required'] = isset($input['consent_required']) ? (int) !!$input['consent_required'] : 0;

		// placements
		foreach (['header','before_content','after_content','in_content','sticky'] as $k) {
			if (!isset($san['placements'][$k])) $san['placements'][$k] = [];
			$san['placements'][$k]['enabled'] = isset($input['placements'][$k]['enabled']) ? (int) !!$input['placements'][$k]['enabled'] : 0;
		}
		$san['placements']['in_content']['paragraph'] = isset($input['placements']['in_content']['paragraph']) ? max(1, (int)$input['placements']['in_content']['paragraph']) : 2;

		// non-amp codes
		foreach (['header_code','before_content_code','after_content_code','in_content_code'] as $k) {
			$san['non_amp'][$k] = isset($input['non_amp'][$k]) ? wp_kses_post($input['non_amp'][$k]) : '';
		}

		// amp
		$san['amp']['header_slot']        = isset($input['amp']['header_slot']) ? sanitize_text_field($input['amp']['header_slot']) : '';
		$san['amp']['in_article_slot']    = isset($input['amp']['in_article_slot']) ? sanitize_text_field($input['amp']['in_article_slot']) : '';
		$san['amp']['after_content_slot'] = isset($input['amp']['after_content_slot']) ? sanitize_text_field($input['amp']['after_content_slot']) : '';
		$san['amp']['sticky_slot']        = isset($input['amp']['sticky_slot']) ? sanitize_text_field($input['amp']['sticky_slot']) : '';
		$san['amp']['width']              = isset($input['amp']['width']) ? max(100, (int)$input['amp']['width']) : 336;
		$san['amp']['height']             = isset($input['amp']['height']) ? max(50, (int)$input['amp']['height']) : 280;

		// post types
		$san['post_types'] = [];
		$post_types = get_post_types(['public' => true], 'names');
		foreach ($post_types as $pt) {
			$san['post_types'][$pt] = isset($input['post_types'][$pt]) ? (int) !!$input['post_types'][$pt] : 0;
		}

		// categories
		$san['excluded_categories'] = isset($input['excluded_categories']) && is_array($input['excluded_categories']) ? array_map('intval', $input['excluded_categories']) : [];

		return $san;
	}

	public function render_page() {
		if ( ! current_user_can('manage_options') ) return;
		$opts = $this->get_options();
		$post_types = get_post_types(['public' => true], 'objects');
		$cats = get_categories(['hide_empty' => false]);
		?>
		<div class="wrap">
			<h1>Pro AMP AdSense Manager</h1>
			<form method="post" action="options.php">
				<?php settings_fields('paam_group'); ?>
				<h2 class="title"><?php _e('Genel Ayarlar', 'pro-amp-adsense'); ?></h2>
				<table class="form-table">
					<tr>
						<th scope="row"><label for="publisher_id">AdSense Publisher ID (ca-pub-...)</label></th>
						<td><input type="text" id="publisher_id" name="paam_options[publisher_id]" value="<?php echo esc_attr($opts['publisher_id'] ?? ''); ?>" class="regular-text" placeholder="ca-pub-xxxxxxxxxxxxxxxx"></td>
					</tr>
					<tr>
						<th scope="row"><label for="consent_required">Onay (Consent) Gerekiyor</label></th>
						<td><label><input type="checkbox" id="consent_required" name="paam_options[consent_required]" value="1" <?php checked( !empty($opts['consent_required']) ); ?>> CMP ile izin alınmadan reklam göstermeyi durdur</label></td>
					</tr>
				</table>

				<h2 class="title"><?php _e('Yerleşimler', 'pro-amp-adsense'); ?></h2>
				<table class="form-table">
					<tr><th>Header</th><td>
						<label><input type="checkbox" name="paam_options[placements][header][enabled]" value="1" <?php checked( !empty($opts['placements']['header']['enabled']) ); ?>> Etkin</label>
					</td></tr>
					<tr><th>İçerikten Önce</th><td>
						<label><input type="checkbox" name="paam_options[placements][before_content][enabled]" value="1" <?php checked( !empty($opts['placements']['before_content']['enabled']) ); ?>> Etkin</label>
					</td></tr>
					<tr><th>İçerik İçi</th><td>
						<label><input type="checkbox" name="paam_options[placements][in_content][enabled]" value="1" <?php checked( !empty($opts['placements']['in_content']['enabled']) ); ?>> Etkin</label>
						&nbsp; Paragraf numarası: <input type="number" min="1" style="width:80px" name="paam_options[placements][in_content][paragraph]" value="<?php echo (int)($opts['placements']['in_content']['paragraph'] ?? 2); ?>">
					</td></tr>
					<tr><th>İçerikten Sonra</th><td>
						<label><input type="checkbox" name="paam_options[placements][after_content][enabled]" value="1" <?php checked( !empty($opts['placements']['after_content']['enabled']) ); ?>> Etkin</label>
					</td></tr>
					<tr><th>Mobil Sticky Alt</th><td>
						<label><input type="checkbox" name="paam_options[placements][sticky][enabled]" value="1" <?php checked( !empty($opts['placements']['sticky']['enabled']) ); ?>> Etkin (AMP: amp-sticky-ad)</label>
					</td></tr>
				</table>

				<h2 class="title"><?php _e('Non‑AMP Reklam Kodları (tam HTML/JS)', 'pro-amp-adsense'); ?></h2>
				<table class="form-table">
					<tr><th>Header</th><td><textarea name="paam_options[non_amp][header_code]" rows="4" class="large-text code"><?php echo esc_textarea($opts['non_amp']['header_code'] ?? ''); ?></textarea></td></tr>
					<tr><th>İçerikten Önce</th><td><textarea name="paam_options[non_amp][before_content_code]" rows="4" class="large-text code"><?php echo esc_textarea($opts['non_amp']['before_content_code'] ?? ''); ?></textarea></td></tr>
					<tr><th>İçerik İçi</th><td><textarea name="paam_options[non_amp][in_content_code]" rows="4" class="large-text code"><?php echo esc_textarea($opts['non_amp']['in_content_code'] ?? ''); ?></textarea></td></tr>
					<tr><th>İçerikten Sonra</th><td><textarea name="paam_options[non_amp][after_content_code]" rows="4" class="large-text code"><?php echo esc_textarea($opts['non_amp']['after_content_code'] ?? ''); ?></textarea></td></tr>
				</table>

				<h2 class="title"><?php _e('AMP Slot Bilgileri', 'pro-amp-adsense'); ?></h2>
				<p>AMP sayfalarında <code>&lt;amp-ad type="adsense"&gt;</code> için slot bilgilerini girin.</p>
				<table class="form-table">
					<tr><th>Header Slot</th><td><input type="text" name="paam_options[amp][header_slot]" value="<?php echo esc_attr($opts['amp']['header_slot'] ?? ''); ?>" class="regular-text"></td></tr>
					<tr><th>İçerik İçi (in-article) Slot</th><td><input type="text" name="paam_options[amp][in_article_slot]" value="<?php echo esc_attr($opts['amp']['in_article_slot'] ?? ''); ?>" class="regular-text"></td></tr>
					<tr><th>İçerikten Sonra Slot</th><td><input type="text" name="paam_options[amp][after_content_slot]" value="<?php echo esc_attr($opts['amp']['after_content_slot'] ?? ''); ?>" class="regular-text"></td></tr>
					<tr><th>Sticky Alt Slot</th><td><input type="text" name="paam_options[amp][sticky_slot]" value="<?php echo esc_attr($opts['amp']['sticky_slot'] ?? ''); ?>" class="regular-text"></td></tr>
					<tr><th>AMP Boyutları</th><td>
						En: <input type="number" min="100" name="paam_options[amp][width]" value="<?php echo (int)($opts['amp']['width'] ?? 336); ?>" style="width:100px">
						Yükseklik: <input type="number" min="50" name="paam_options[amp][height]" value="<?php echo (int)($opts['amp']['height'] ?? 280); ?>" style="width:100px">
					</td></tr>
				</table>

				<h2 class="title"><?php _e('Hedefleme', 'pro-amp-adsense'); ?></h2>
				<table class="form-table">
					<tr><th>Yayın Tipleri</th><td>
						<?php foreach ($post_types as $pt) : ?>
							<label style="margin-right:15px"><input type="checkbox" name="paam_options[post_types][<?php echo esc_attr($pt->name); ?>]" value="1" <?php checked( !empty($opts['post_types'][$pt->name]) ); ?>> <?php echo esc_html($pt->labels->singular_name); ?></label>
						<?php endforeach; ?>
					</td></tr>
					<tr><th>Hariç Kategoriler</th><td style="max-width:600px">
						<select name="paam_options[excluded_categories][]" multiple size="8" style="min-width: 320px">
							<?php foreach ($cats as $cat) : ?>
								<option value="<?php echo (int)$cat->term_id; ?>" <?php selected( in_array($cat->term_id, $opts['excluded_categories'] ?? []) ); ?>>
									<?php echo esc_html($cat->name); ?>
								</option>
							<?php endforeach; ?>
						</select>
						<p class="description">Seçilen kategorilerde reklam gösterilmez.</p>
					</td></tr>
				</table>

				<?php submit_button(); ?>
			</form>

			<hr>
			<h2>Kısa Kod</h2>
			<p>İstediğiniz yerde reklam göstermek için: <code>[paam_ad place="after_content"]</code>. Parametreler: <code>header, before_content, in_content, after_content, sticky</code></p>
			<p>Widget: Görünüm → Bileşenler → <strong>Pro AMP AdSense Widget</strong></p>
		</div>
		<?php
	}
}
